<?php
require_once 'auth.php';
require_once 'config.php';

if (!Auth::isLoggedIn() || !Auth::isAdmin()) {
    header('Location: admin_login.php');
    exit;
}

$db = Config::getDB();
$action = $_GET['action'] ?? 'list';
$user_id = $_GET['user_id'] ?? 0;

$success = '';
$error = '';

// Handle form submissions
if (($_POST['action'] ?? '') === 'add_user') {
    $username = Config::sanitize($_POST['username']);
    $email = Config::sanitize($_POST['email']);
    $phoneNumber = Config::sanitize($_POST['phoneNumber'] ?? '');
    $course_sel = !empty($_POST['course_sel']) ? intval($_POST['course_sel']) : null;
    $user_type = Config::sanitize($_POST['user_type']);
    $password = $_POST['password'] ?? 'password123';
    
    $password_hash = password_hash($password, PASSWORD_DEFAULT);
    
    try {
        $stmt = $db->prepare("INSERT INTO users (username, email, password_hash, user_type, phoneNumber, course_sel) VALUES (?, ?, ?, ?, ?, ?)");
        if ($stmt->execute([$username, $email, $password_hash, $user_type, $phoneNumber, $course_sel])) {
            $success = "User created successfully! Default password: " . htmlspecialchars($password);
        }
    } catch (PDOException $e) {
        $error = "Failed to create user: " . $e->getMessage();
    }
}

if (($_POST['action'] ?? '') === 'edit_user') {
    $username = Config::sanitize($_POST['username']);
    $email = Config::sanitize($_POST['email']);
    $phoneNumber = Config::sanitize($_POST['phoneNumber'] ?? '');
    $course_sel = !empty($_POST['course_sel']) ? intval($_POST['course_sel']) : null;
    $user_type = Config::sanitize($_POST['user_type']);
    
    // Handle password change if provided
    if (!empty($_POST['password'])) {
        $password_hash = password_hash($_POST['password'], PASSWORD_DEFAULT);
        $stmt = $db->prepare("UPDATE users SET username = ?, email = ?, password_hash = ?, user_type = ?, phoneNumber = ?, course_sel = ? WHERE id = ?");
        $result = $stmt->execute([$username, $email, $password_hash, $user_type, $phoneNumber, $course_sel, $user_id]);
    } else {
        $stmt = $db->prepare("UPDATE users SET username = ?, email = ?, user_type = ?, phoneNumber = ?, course_sel = ? WHERE id = ?");
        $result = $stmt->execute([$username, $email, $user_type, $phoneNumber, $course_sel, $user_id]);
    }
    
    if ($result) {
        $success = "User updated successfully!";
    } else {
        $error = "Failed to update user.";
    }
}

if (($_POST['action'] ?? '') === 'bulk_upload' && isset($_FILES['users_file'])) {
    if ($_FILES['users_file']['error'] === UPLOAD_ERR_OK) {
        $bulk_course_id = intval($_POST['bulk_course_id'] ?? 0);
            
        if ($bulk_course_id === 0) {
            $error = "Please select a course for the users.";
        } else {
            // echo $bulk_course_id;
           $upload_result = handleBulkUpload($_FILES['users_file'], $bulk_course_id);
            if ($upload_result['success']) {
                $success = "Successfully uploaded " . $upload_result['count'] . " users to the selected course!";
                if (!empty($upload_result['errors'])) {
                    $error = "Some users failed: " . implode(', ', array_slice($upload_result['errors'], 0, 5));
                }
            } else {
                $error = "Upload failed: " . $upload_result['error'];
            }
        }
    } else {
        $error = "File upload error: " . $_FILES['users_file']['error'];
    }
}

// Handle user actions
if (($_GET['action'] ?? '') === 'delete' && $user_id) {
    // Prevent deleting admin users
    $stmt = $db->prepare("SELECT user_type FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($user && $user['user_type'] !== 'admin') {
        $stmt = $db->prepare("DELETE FROM users WHERE id = ?");
        if ($stmt->execute([$user_id])) {
            $success = "User deleted successfully!";
        }
    } else {
        $error = "Cannot delete admin users.";
    }
}

// Get all users with course information
try {
    $stmt = $db->prepare("
        SELECT u.*, c.id as course_id, c.crs_title, c.course_name 
        FROM users u 
        LEFT JOIN courses c ON u.course_sel = c.id 
        ORDER BY u.created_at DESC
    ");
    $stmt->execute();
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    // Fallback if join fails
    $stmt = $db->prepare("SELECT * FROM users ORDER BY created_at DESC");
    $stmt->execute();
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Get specific user for editing
$user = null;
if ($user_id) {
    $stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
}

// Get courses for dropdown
try {
    $stmt = $db->prepare("
    SELECT id, crs_title, course_name 
    FROM courses 
    WHERE id IN (1,2,3,5,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,29,32,35,36,37,48,49,58,65,69) 
    ORDER BY crs_title
");
    $stmt->execute();
    $courses = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $courses = [];
}

function handleBulkUpload($file, $bulk_course_id) {
    $db = Config::getDB();
    $allowed_types = ['text/csv', 'application/vnd.ms-excel', 'text/plain'];
    
    if (!in_array($file['type'], $allowed_types)) {
        return ['success' => false, 'error' => 'Invalid file type. Please upload CSV file.'];
    }
    
    $handle = fopen($file['tmp_name'], 'r');
    if (!$handle) {
        return ['success' => false, 'error' => 'Cannot open uploaded file.'];
    }
    
    $count = 0;
    $errors = [];
    $first_row = true;
    
    while (($data = fgetcsv($handle, 1000, ',')) !== FALSE) {
        if ($first_row) {
            $first_row = false;
            continue; // Skip header row
        }
        
        if (count($data) < 2) {
            $errors[] = "Row " . ($count + 1) . ": Insufficient data";
            continue;
        }
        
        $username = Config::sanitize($data[0] ?? '');
        $email = Config::sanitize($data[1] ?? '');
        $user_type = Config::sanitize($data[2] ?? 'student');
        $phoneNumber = Config::sanitize($data[3] ?? '');
        
        if (empty($username) || empty($email)) {
            $errors[] = "Row " . ($count + 1) . ": Missing username or email";
            continue;
        }
        
        $password_hash = password_hash('password123', PASSWORD_DEFAULT); // Default password
        
        try {
            // echo $bulk_course_id;
            $stmt = $db->prepare("INSERT INTO users (username, email, password_hash, user_type, phoneNumber, course_sel) VALUES (?, ?, ?, ?, ?, ?)");
            if ($stmt->execute([$username, $email, $password_hash, $user_type, $phoneNumber, $bulk_course_id])) {
                $count++;
            } else {
                $errors[] = "Row " . ($count + 1) . ": Insertion failed - " . $username;
            }
        } catch (PDOException $e) {
            if (strpos($e->getMessage(), 'Duplicate entry') !== false) {
                $errors[] = "Row " . ($count + 1) . ": Username or email already exists - " . $username;
            } else {
                $errors[] = "Row " . ($count + 1) . ": " . $e->getMessage();
            }
        }
    }
    
    fclose($handle);
    
    return [
        'success' => true, 
        'count' => $count, 
        'errors' => $errors
    ];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>User Management - Admin</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.datatables.net/1.13.6/css/dataTables.bootstrap5.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <nav class="navbar navbar-expand-lg navbar-dark bg-dark">
        <div class="container">
            <a class="navbar-brand" href="#">CBT System - Admin</a>
            <div class="navbar-nav ms-auto">
                <a href="admin_dashboard.php" class="nav-link">Dashboard</a>
                <a href="admin_exams.php" class="nav-link">Exams</a>
                <a href="admin_results.php" class="nav-link">Results</a>
                <a href="logout.php" class="nav-link">Logout</a>
            </div>
        </div>
    </nav>

    <div class="container mt-4">
        <!-- Notifications -->
        <?php if (isset($success)): ?>
            <div class="alert alert-success"><?= $success ?></div>
        <?php endif; ?>
        <?php if (isset($error)): ?>
            <div class="alert alert-danger"><?= $error ?></div>
        <?php endif; ?>

        <div class="row">
            <div class="col-12">
                <!-- Navigation Tabs -->
                <ul class="nav nav-tabs mb-4">
                    <li class="nav-item">
                        <a class="nav-link <?= $action === 'list' ? 'active' : '' ?>" href="?action=list">All Users</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link <?= in_array($action, ['add_user', 'edit_user']) ? 'active' : '' ?>" 
                           href="?action=add_user"><?= $user_id ? 'Edit User' : 'Add User' ?></a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link <?= $action === 'bulk_upload' ? 'active' : '' ?>" 
                           href="?action=bulk_upload">Bulk Upload</a>
                    </li>
                </ul>

                <!-- Content Sections -->
                <?php if ($action === 'list'): ?>
                    <!-- Users List -->
                    <div class="card">
                        <div class="card-header d-flex justify-content-between align-items-center">
                            <h4 class="mb-0"><i class="fas fa-users me-2"></i>User Management</h4>
                            <div>
                                <a href="?action=add_user" class="btn btn-primary btn-sm">
                                    <i class="fas fa-plus me-1"></i>Add User
                                </a>
                                <a href="?action=bulk_upload" class="btn btn-success btn-sm">
                                    <i class="fas fa-upload me-1"></i>Bulk Upload
                                </a>
                            </div>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-striped" id="usersTable">
                                    <thead>
                                        <tr>
                                            <th>ID</th>
                                            <th>Username</th>
                                            <th>Email</th>
                                            <th>Phone</th>
                                            <th>Course</th>
                                            <th>Type</th>
                                            <th>Created</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($users as $user): ?>
                                        <tr>
                                            <td><?= $user['id'] ?></td>
                                            <td><?= htmlspecialchars($user['username']) ?></td>
                                            <td><?= htmlspecialchars($user['email']) ?></td>
                                            <td><?= htmlspecialchars($user['phoneNumber'] ?? 'N/A') ?></td>
                                            <td>
                                                <?php if (!empty($user['crs_title'])): ?>
                                                    <span class="badge bg-info" title="<?= htmlspecialchars($user['course_name'] ?? '') ?>">
                                                        <?= htmlspecialchars($user['crs_title']) ?>
                                                    </span>
                                                <?php elseif (!empty($user['course_name'])): ?>
                                                    <span class="badge bg-info">
                                                        <?= htmlspecialchars($user['course_name']) ?>
                                                    </span>
                                                <?php else: ?>
                                                    <em class="text-muted">Not set</em>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <span class="badge bg-<?= $user['user_type'] === 'admin' ? 'danger' : 'primary' ?>">
                                                    <?= ucfirst($user['user_type']) ?>
                                                </span>
                                            </td>
                                            <td><?= date('M j, Y', strtotime($user['created_at'])) ?></td>
                                            <td>
                                                <div class="btn-group btn-group-sm">
                                                    <a href="?action=edit_user&user_id=<?= $user['id'] ?>" 
                                                       class="btn btn-warning" title="Edit">
                                                        <i class="fas fa-edit"></i>
                                                    </a>
                                                    <?php if ($user['user_type'] !== 'admin'): ?>
                                                        <a href="?action=delete&user_id=<?= $user['id'] ?>" 
                                                           class="btn btn-danger" 
                                                           title="Delete"
                                                           onclick="return confirm('Are you sure you want to delete this user?')">
                                                            <i class="fas fa-trash"></i>
                                                        </a>
                                                    <?php endif; ?>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>

                <?php elseif (in_array($action, ['add_user', 'edit_user'])): ?>
                    <!-- Add/Edit User Form -->
                    <div class="card">
                        <div class="card-header">
                            <h4 class="mb-0">
                                <i class="fas fa-<?= $user_id ? 'edit' : 'user-plus' ?> me-2"></i>
                                <?= $user_id ? 'Edit User' : 'Add New User' ?>
                            </h4>
                        </div>
                        <div class="card-body">
                            <form method="POST">
                                <input type="hidden" name="action" value="<?= $user_id ? 'edit_user' : 'add_user' ?>">
                                <?php if ($user_id): ?>
                                    <input type="hidden" name="user_id" value="<?= $user_id ?>">
                                <?php endif; ?>
                                
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label class="form-label">Username *</label>
                                            <input type="text" name="username" class="form-control" 
                                                   value="<?= $user['username'] ?? '' ?>" required>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label class="form-label">Email *</label>
                                            <input type="email" name="email" class="form-control" 
                                                   value="<?= $user['email'] ?? '' ?>" required>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label class="form-label">Phone Number</label>
                                            <input type="tel" name="phoneNumber" class="form-control" 
                                                   value="<?= $user['phoneNumber'] ?? '' ?>">
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label class="form-label">Course</label>
                                            <select name="course_sel" class="form-select">
                                                <option value="">Select Course...</option>
                                                <?php foreach ($courses as $course): ?>
                                                    <option value="<?= $course['id'] ?>" 
                                                        <?= ($user['course_sel'] ?? '') == $course['id'] ? 'selected' : '' ?>>
                                                        <?= htmlspecialchars($course['crs_title'] . ' - ' . ($course['course_name'] ?? '')) ?>
                                                    </option>
                                                <?php endforeach; ?>
                                            </select>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label class="form-label">User Type *</label>
                                            <select name="user_type" class="form-select" required>
                                                <option value="student" <?= ($user['user_type'] ?? '') === 'student' ? 'selected' : '' ?>>Student</option>
                                                <option value="admin" <?= ($user['user_type'] ?? '') === 'admin' ? 'selected' : '' ?>>Admin</option>
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label class="form-label">Password <?= $user_id ? '(leave blank to keep current)' : '*' ?></label>
                                            <input type="password" name="password" class="form-control" 
                                                   <?= !$user_id ? 'required' : '' ?> 
                                                   placeholder="<?= $user_id ? 'Enter new password' : 'Set password' ?>">
                                            <?php if (!$user_id): ?>
                                                <small class="form-text text-muted">Default: password123</small>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="d-flex gap-2">
                                    <button type="submit" class="btn btn-primary">
                                        <i class="fas fa-save me-1"></i>
                                        <?= $user_id ? 'Update User' : 'Create User' ?>
                                    </button>
                                    <a href="?action=list" class="btn btn-secondary">Cancel</a>
                                </div>
                            </form>
                        </div>
                    </div>

                <?php elseif ($action === 'bulk_upload'): ?>
                    <!-- Bulk Upload -->
                    <div class="card">
                        <div class="card-header">
                            <h4 class="mb-0">
                                <i class="fas fa-upload me-2"></i>
                                Bulk Upload Users
                            </h4>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <div class="col-md-6">
                                    <form method="POST" enctype="multipart/form-data">
                                        <input type="hidden" name="action" value="bulk_upload">
                                        
                                        <div class="mb-3">
                                            <label class="form-label">Select Course for All Users *</label>
                                            <select name="bulk_course_id" class="form-select" required>
                                                <option value="">Choose a course...</option>
                                                <?php foreach ($courses as $course):  ?>
                                                    <option value="<?= $course['id'] ?>" <?= ($_POST['bulk_course_id'] ?? '') == $course['id'] ? 'selected' : '' ?>>
                                                        <?= htmlspecialchars($course['crs_title'] ) ?>
                                                    </option>
                                                <?php endforeach; ?>
                                            </select>
                                            <small class="form-text text-muted">
                                                All users in the CSV will be assigned to this course
                                            </small>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label class="form-label">Upload CSV File *</label>
                                            <input type="file" name="users_file" class="form-control" accept=".csv,.xlsx,.xls" required>
                                            <small class="form-text text-muted">
                                                File must be in CSV format with appropriate columns
                                            </small>
                                        </div>
                                        
                                        <button type="submit" class="btn btn-success">
                                            <i class="fas fa-upload me-1"></i>Upload Users
                                        </button>
                                        <a href="?action=list" class="btn btn-secondary">Cancel</a>
                                    </form>
                                </div>
                                <div class="col-md-6">
                                    <div class="alert alert-info">
                                        <h6><i class="fas fa-info-circle me-2"></i>CSV Format Requirements</h6>
                                        
                                        <table class="table table-sm mt-2">
                                            <thead>
                                                <tr>
                                                    <th>Username</th>
                                                    <th>Email</th>
                                                    <th>User Type</th>
                                                    <th>Phone Number</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <tr>
                                                    <td>john_doe</td>
                                                    <td>john@email.com</td>
                                                    <td>student</td>
                                                    <td>+1234567890</td>
                                                </tr>
                                                <tr>
                                                    <td>jane_smith</td>
                                                    <td>jane@email.com</td>
                                                    <td>student</td>
                                                    <td>+1234567891</td>
                                                </tr>
                                            </tbody>
                                        </table>
                                        
                                        <p class="mb-2"><strong>Notes:</strong></p>
                                        <ul class="mb-0">
                                            <li>First row should be headers</li>
                                            <li>Username and Email are required</li>
                                            <li>User Type defaults to "student" if not specified</li>
                                            <li>All users will be assigned to the selected course above</li>
                                            <li>All users get default password: <strong>password123</strong></li>
                                        </ul>
                                    </div>
                                    
                                    <div class="text-center">
                                        <a href="sample_users.csv" class="btn btn-outline-primary btn-sm" download>
                                            <i class="fas fa-download me-1"></i>Download Sample CSV
                                        </a>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.6/js/dataTables.bootstrap5.min.js"></script>
    <script>
        $(document).ready(function() {
            $('#usersTable').DataTable({
                pageLength: 25,
                order: [[0, 'desc']],
                dom: '<"row"<"col-md-6"l><"col-md-6"f>>rt<"row"<"col-md-6"i><"col-md-6"p>>'
            });
        });
    </script>
</body>
</html>